<?php
$data = json_decode(shell_exec("bash server_status.sh"), true);

// Clean up PHP version string (remove OS suffixes like -0ubuntu0.18.04.15)
if (!empty($data['php_version'])) {
    $data['php_version'] = preg_replace('/-.*/', '', $data['php_version']);
}

// Clean MySQL version string
if (!empty($data['mysql_version'])) {
    if (preg_match('/^(\d+\.\d+\.\d+-\d+)/', $data['mysql_version'], $matches)) {
        $data['mysql_version'] = $matches[1];
    }
}

// Get hostname and IP
$hostname = gethostname();
$ipAddress = $_SERVER['SERVER_ADDR'] ?? trim(shell_exec("hostname -I | awk '{print $1}'"));

// Helper functions
function versionCompare($current, $latest) {
    if (!$current || !$latest) return 'Warning';
    return version_compare($current, $latest, '>=') ? 'Good' : 'Critical';
}

function statusColor($status) {
    return ['Good'=>'green','Warning'=>'orange','Critical'=>'red'][$status] ?? 'grey';
}

function osVersionStatus($osName, $currentVersion) {
    $latestVersions = ['Ubuntu'=>'24.04','Debian'=>'12','CentOS'=>'9','Rocky'=>'9','AlmaLinux'=>'9'];
    $latest = $latestVersions[$osName] ?? null;
    if (!$latest) return 'Warning';
    return version_compare($currentVersion,$latest,'>=')?'Good':'Critical';
}

function diskStatus($percent){return $percent>=90?'Critical':($percent>=70?'Warning':'Good');}
function memStatus($percent){return $percent>=90?'Critical':($percent>=70?'Warning':'Good');}
function cpuStatus($load){return $load>=4?'Critical':($load>=2?'Warning':'Good');}

// Convert uptime string to total days
function uptimeToDays($uptimeStr) {
    $weeks = $days = $hours = $minutes = 0;
    if (preg_match('/(\d+)\s*week/', $uptimeStr, $m)) $weeks = (int)$m[1];
    if (preg_match('/(\d+)\s*day/', $uptimeStr, $m))  $days = (int)$m[1];
    if (preg_match('/(\d+)\s*hour/', $uptimeStr, $m)) $hours = (int)$m[1];
    if (preg_match('/(\d+)\s*minute/', $uptimeStr, $m)) $minutes = (int)$m[1];
    $totalDays = ($weeks * 7) + $days + ($hours / 24) + ($minutes / 1440);
    return round($totalDays, 1);
}

// Convert uptime string and determine status
$totalUptimeDays = uptimeToDays($data['uptime']);
if ($totalUptimeDays >= 30) {
    $uptimeStatus = 'Good';
} elseif ($totalUptimeDays >= 7) {
    $uptimeStatus = 'Warning';
} else {
    $uptimeStatus = 'Critical';
}

$osStatus = osVersionStatus($data['os_name'],$data['os_version']);

$checks = [
    'Operating System'=>['value'=>$data['os_name'].' '.$data['os_version'],'status'=>$osStatus,'icon'=>'🖥️'],
    'Uptime' => ['value' => "{$totalUptimeDays} days", 'status' => $uptimeStatus, 'icon' => '⏱️'],
    'PHP Version'=>['value'=>$data['php_version'],'status'=>versionCompare($data['php_version'],$data['latest_php']),'icon'=>'🐘'],
    'MySQL Version'=>['value'=>$data['mysql_version'],'status'=>versionCompare($data['mysql_version'],$data['latest_mysql']),'icon'=>'🛢️'],
    'Apache Version'=>['value'=>$data['apache_version'],'status'=>versionCompare($data['apache_version'],$data['latest_apache']),'icon'=>'🌐'],
    'SSH Service'=>['value'=>ucfirst($data['ssh_status']),'status'=>$data['ssh_status']==='active'?'Good':'Critical','icon'=>'🔒'],
    'Disk Usage'=>['value'=>$data['disk_usage'],'status'=>diskStatus($data['disk_percent']),'icon'=>'💾'],
    'Memory Usage'=>['value'=>$data['mem_usage'],'status'=>memStatus($data['mem_percent']),'icon'=>'🧠'],
    'CPU Load'=>['value'=>$data['cpu_load'],'status'=>cpuStatus($data['cpu_load']),'icon'=>'⚡'],
    'APT Updates'=>['value'=>$data['apt_updates'].' pending','status'=>$data['apt_updates']>0?'Warning':'Good','icon'=>'📦']
];

// --- HTML Output ---
ob_start();
?>
<div style="max-width:700px; margin:auto; font-family:Arial,sans-serif;">
    <!-- Hostname & IP block -->
    <div style="display:flex; gap:30px; background:#f0f0f0; padding:12px 15px; border-radius:12px; margin-bottom:15px; border:1px solid #ddd;">
        <div><strong>🌐 Hostname:</strong> <?= htmlspecialchars($hostname) ?></div>
        <div><strong>💻 IP Address:</strong> <?= htmlspecialchars($ipAddress) ?></div>
    </div>

    <!-- Server status table -->
    <div style="border-radius:12px; overflow:hidden; border:1px solid #ddd;">
        <table style="width:100%; border-collapse:collapse;">
            <tr style="background:#0078d7; color:#fff; text-align:left;">
                <th style="padding:12px 15px; border-top-left-radius:12px;">Service</th>
                <th style="padding:12px 15px;">Current</th>
                <th style="padding:12px 15px; border-top-right-radius:12px;">Status</th>
            </tr>
            <?php foreach($checks as $service=>$info):
                $colorClass = statusColor($info['status']);
                $bgColor = $colorClass==='green'?'#d4edda':($colorClass==='orange'?'#fff3cd':'#f8d7da');
                $textColor = $colorClass==='green'?'#155724':($colorClass==='orange'?'#856404':'#721c24');
            ?>
            <tr style="border-bottom:1px solid #ddd;">
                <td style="padding:10px; vertical-align:middle;"><?= $info['icon'] ?> <?= htmlspecialchars($service) ?></td>
                <td style="padding:10px; vertical-align:middle;"><?= htmlspecialchars($info['value']) ?></td>
                <td style="padding:10px; vertical-align:middle; text-align:center;">
                    <span style="display:inline-block; padding:5px 12px; border-radius:12px; background:<?= $bgColor ?>; color:<?= $textColor ?>; font-weight:bold;">
                        <?= htmlspecialchars($info['status']) ?>
                    </span>
                </td>
            </tr>
            <?php endforeach; ?>
        </table>
    </div>
</div>
<?php
$htmlContent = ob_get_clean();

// Email alert thresholds
$alert = false;
if(intval($data['disk_percent'])>70 || intval($data['mem_percent'])>70 || floatval($data['cpu_load'])>2) $alert=true;

/* --- Optional Email Alert ---
if($alert){
    $to="admin@example.com";
    $subject="🚨 Server Health Alert - {$hostname}";
    $headers="From: monitor@example.com\r\nMIME-Version:1.0\r\nContent-Type:text/html;charset=UTF-8\r\n";
    mail($to,$subject,$htmlContent,$headers);
}
*/

// Output dashboard HTML
echo $htmlContent;
?>
