#!/bin/bash

# Collect system info dynamically
OS_NAME=$(lsb_release -si 2>/dev/null || echo "Unknown")
OS_VERSION=$(lsb_release -sr 2>/dev/null || echo "Unknown")
PHP_VERSION=$(php -r 'echo PHP_VERSION;' 2>/dev/null)
MYSQL_VERSION=$(mysql -V 2>/dev/null | awk '{for(i=1;i<=NF;i++){if($i~/Ver/){print $(i+1);exit}}}' | sed 's/,//')
APACHE_VERSION=$(apache2 -v 2>/dev/null | grep "Server version" | awk '{print $3}' | cut -d'/' -f2)
SSH_STATUS=$(systemctl is-active ssh 2>/dev/null)
APT_UPDATES=$(apt list --upgradable 2>/dev/null | grep -c upgradable)

# System uptime
UPTIME=$(uptime -p)

# Disk usage for root
DISK_USAGE_RAW=$(df -h / | awk 'NR==2 {print $3 "/" $2 " (" $5 ")"}')
DISK_PERCENT=$(df / | awk 'NR==2 {gsub("%",""); print $5}')

# Memory usage in GB
MEM_TOTAL_MB=$(free -m | awk '/Mem:/ {print $2}')
MEM_USED_MB=$(free -m | awk '/Mem:/ {print $3}')
MEM_TOTAL_GB=$(awk "BEGIN {printf \"%.2f\", $MEM_TOTAL_MB/1024}")
MEM_USED_GB=$(awk "BEGIN {printf \"%.2f\", $MEM_USED_MB/1024}")
MEM_USAGE="$MEM_USED_GB GB / $MEM_TOTAL_GB GB"
MEM_PERCENT=$(( MEM_USED_MB*100/MEM_TOTAL_MB ))

# CPU load (1 min)
CPU_LOAD=$(uptime | awk -F'load average:' '{print $2}' | cut -d, -f1 | xargs)

# Fetch latest versions dynamically (best-effort)
LATEST_PHP=$(curl -s https://www.php.net/releases/index.php | grep -m1 -oP 'PHP \K[0-9]+\.[0-9]+\.[0-9]+')
LATEST_MYSQL=$(curl -s https://dev.mysql.com/downloads/mysql/ | grep -m1 -oP 'MySQL Community Server \K[0-9]+\.[0-9]+\.[0-9]+')
LATEST_APACHE=$(curl -s https://httpd.apache.org/download.cgi | grep -m1 -oP 'Apache HTTP Server \K[0-9]+\.[0-9]+\.[0-9]+')

# Output JSON
cat <<EOF
{
  "os_name": "$OS_NAME",
  "os_version": "$OS_VERSION",
  "php_version": "$PHP_VERSION",
  "latest_php": "$LATEST_PHP",
  "mysql_version": "$MYSQL_VERSION",
  "latest_mysql": "$LATEST_MYSQL",
  "apache_version": "$APACHE_VERSION",
  "latest_apache": "$LATEST_APACHE",
  "ssh_status": "$SSH_STATUS",
  "apt_updates": "$APT_UPDATES",
  "disk_usage": "$DISK_USAGE_RAW",
  "disk_percent": "$DISK_PERCENT",
  "mem_usage": "$MEM_USAGE",
  "mem_percent": "$MEM_PERCENT",
  "cpu_load": "$CPU_LOAD",
  "uptime": "$UPTIME"
}
EOF
